import { useState, useEffect, useMemo } from '@wordpress/element';

import { Spinner } from '@newfold/ui-component-library';
import {
	ShieldCheckIcon,
	ExclamationCircleIcon,
	ExclamationTriangleIcon,
} from '@heroicons/react/24/outline';

import getMalwareCheckText from './getMalwareCheckText';
import SiteStatusCard from '../SiteStatusCard';

const MalwareCheckCard = ( { methods, data } ) => {
	const [ isLoading, setIsLoading ] = useState( true );
	const [ scanStatus, setScanStatus ] = useState( 'unknown' );
	const [ numThreats, setNumThreats ] = useState( 0 );
	const [ isRegistered, setIsRegistered ] = useState( false );
	const [ message, setMessage ] = useState(
		getMalwareCheckText().messages.scanning
	);

	const registrationNonce = data?.registration_nonce;
	const api = data?.api || {};
	const urls = data?.urls || {};
	const installToken = data?.install_token || '';
	const plugin = data?.plugin || 'jetpack-protect';
	const cachedStatus = data?.scan_status || 'unknown';

	useEffect( () => {
		if ( cachedStatus !== 'unknown' ) {
			setScanStatus( cachedStatus );
			setIsLoading( false );
			return;
		}

		const installRegisterAndScan = async () => {
			try {
				// Install and Activate Jetpack Protect
				await methods.apiFetch( {
					url: api.pluginInstall,
					method: 'POST',
					headers: { 'X-NFD-INSTALLER': installToken },
					data: {
						plugin,
						activate: true,
						queue: false,
					},
				} );

				// Register Jetpack Connection (Only if we have a nonce)
				if ( registrationNonce ) {
					const registerResponse = await methods.apiFetch( {
						path: api.jetpackRegister,
						method: 'POST',
						data: {
							registration_nonce: registrationNonce,
							no_iframe: true,
							plugin_slug: 'connection_package',
							from: 'protect',
						},
					} );
					setIsRegistered( !! registerResponse?.authorizeUrl );
				}

				// Fetch Current Scan Status
				const scanStatusData = await methods.apiFetch( {
					path: api.scanStatus,
					method: 'GET',
				} );

				if ( ! scanStatusData || scanStatusData.error ) {
					throw new Error( getMalwareCheckText().messages.apiFailed );
				}

				const finalStatus =
					scanStatusData.num_threats > 0
						? 'partial-unsafe'
						: 'partial-safe';
				setScanStatus( finalStatus );
				await methods.apiFetch( {
					url: methods.NewfoldRuntime.createApiUrl(
						'/newfold-hosting/v1/panel/update'
					),
					method: 'POST',
					data: {
						identifier: 'malware-check',
						action: 'update_scan_status',
						data: {
							scan_status: finalStatus,
						},
					},
				} );
				setNumThreats( scanStatusData.num_threats );
			} catch ( error ) {
				setScanStatus( 'unknown' );
				setMessage( getMalwareCheckText().messages.error );
			} finally {
				setIsLoading( false );
			}
		};

		installRegisterAndScan();
	}, [ methods, registrationNonce, api, installToken ] );

	const { status, descriptions, buttons, title } = getMalwareCheckText();

	// Use memoized values to avoid unnecessary re-renders
	const statusText = useMemo(
		() => status[ scanStatus ] || status.unknown,
		[ scanStatus, status ]
	);
	const description = useMemo(
		() => descriptions[ scanStatus ] || descriptions.unknown,
		[ scanStatus, descriptions ]
	);

	const Illustration = useMemo( () => {
		switch ( scanStatus ) {
			case 'partial-safe':
				return (
					<ShieldCheckIcon
						width={ 48 }
						height={ 48 }
						className="nfd-text-green-500"
					/>
				);
			case 'partial-unsafe':
				return (
					<ExclamationCircleIcon
						width={ 48 }
						height={ 48 }
						className="nfd-text-red-500"
					/>
				);
			default:
				return (
					<ExclamationTriangleIcon
						width={ 48 }
						height={ 48 }
						className="nfd-text-gray-500"
					/>
				);
		}
	}, [ scanStatus ] );

	return (
		<SiteStatusCard
			title={ title }
			status={ isLoading ? buttons.scanning : statusText }
			description={ isLoading ? '' : description }
			primaryButtonText={ buttons.runScan }
			primaryButtonAction={ () => window.open( urls.scanPage, '_blank' ) }
			primaryButtonDisabled={ isLoading }
			primaryButtonContent={
				isLoading ? (
					<>
						<Spinner />
						<span className="nfd-ml-2">{ message }</span>
					</>
				) : null
			}
			secondaryButtonText={
				numThreats > 0 ? buttons.contactSupport : null
			}
			secondaryButtonAction={
				numThreats > 0
					? () => ( window.location.href = urls.supportPage )
					: null
			}
			linkHref={ isRegistered ? urls.scanPage : '#' }
			linkText={ isRegistered ? buttons.details : '' }
			Illustration={ () => Illustration }
		/>
	);
};

export default MalwareCheckCard;
